import copy
from components.episode_buffer import EpisodeBatch
from modules.mixers.multi_task.vdn import VDNMixer
from modules.mixers.qmix import QMixer
from modules.mixers.multi_task.qattn import QMixer as MTAttnQMixer
import torch as th
from torch.optim import RMSprop, Adam
import torch.nn.functional as F
import math

import os
from copy import deepcopy
import random
import torch.nn as nn
from modules.agents.multi_task.vqvae_agent import VQVAEAgent

class BiKTLearner:
    def __init__(self, mac, logger, main_args,):
        self.main_args = main_args
        self.mac = mac
        self.logger = logger
        

        # get some attributes from mac
        self.task2args = mac.task2args
        self.task2n_agents = mac.task2n_agents
        self.surrogate_decomposer = mac.surrogate_decomposer
        self.task2decomposer = mac.task2decomposer

        self.params = list(mac.parameters())

        self.last_target_update_episode = 0

        self.mixer = None
        if main_args.mixer is not None:
            if main_args.mixer == "vdn":
                self.mixer = VDNMixer()
            elif main_args.mixer == "mt_qattn":
                self.mixer = MTAttnQMixer(self.surrogate_decomposer, main_args)
            else:
                raise ValueError(f"Mixer {main_args.mixer} not recognised.")
            self.params += list(self.mixer.parameters())
            self.target_mixer = copy.deepcopy(self.mixer)

        # self._reset_optimizer()

        # a little wasteful to deepcopy (e.g. duplicates action selector), but should work for any MAC
        self.target_mac = copy.deepcopy(mac)

        # define attributes for each specific task
        self.task2train_info, self.task2encoder_params, self.task2encoder_optimiser = {}, {}, {}
        self.task2traindt_info = {}
        for task in self.task2args:
            task_args = self.task2args[task]
            self.task2train_info[task] = {}
            self.task2train_info[task]["log_stats_t"] = -task_args.learner_log_interval - 1
            
            self.task2traindt_info[task] = {}
            self.task2traindt_info[task]["log_stats_t"] = -task_args.learner_log_interval - 1

        self.c = main_args.c_step
        self.skill_dim = main_args.skill_dim
        self.beta = main_args.beta
        self.alpha = main_args.coef_conservative
        self.phi = main_args.coef_dist

        self.pretrain_steps = 0
        self.training_steps = 0


        self.mini_batch_size = 5
        
        if self.main_args.train_DT:
            vae_path = main_args.pretrain_vae_model_checkpoint
            self.load_models(vae_path)
            self.logger.console_logger.info("Load pretrained models from {}".format(vae_path))
        
        elif self.main_args.train_DT_w_glsk:
            vae_path = main_args.pretrain_vae_model_checkpoint
            self.load_models(vae_path)
            self.logger.console_logger.info("Load pretrained models from {}".format(vae_path))
        
        self.vqvae_beta = self.main_args.vqvae_beta
        
                
    def _reset_optimizer(self):
        if self.main_args.optim_type.lower() == "rmsprop":
            self.pre_optimiser = RMSprop(params=self.params, lr=self.main_args.lr, alpha=self.main_args.optim_alpha, eps=self.main_args.optim_eps, weight_decay=self.main_args.weight_decay)
            self.optimiser = RMSprop(params=self.params, lr=self.main_args.lr, alpha=self.main_args.optim_alpha, eps=self.main_args.optim_eps, weight_decay=self.main_args.weight_decay)
        elif self.main_args.optim_type.lower() == "adam":
            self.pre_optimiser = Adam(params=self.params, lr=self.main_args.lr, weight_decay=self.main_args.weight_decay)
            self.optimiser = Adam(params=self.params, lr=self.main_args.critic_lr, weight_decay=self.main_args.weight_decay)
        else:
            raise ValueError("Invalid optimiser type", self.main_args.optim_type)
        self.pre_optimiser.zero_grad()
        self.optimiser.zero_grad()
    
    def _reset_optimizer_4dt(self, logger):
        if self.main_args.optim_type.lower() == "adam":
            self.optimiser_4dt = Adam(params=self.mac.dt_agent.parameters(), lr=self.main_args.dt_lr, weight_decay=self.main_args.weight_decay)
            self.optimiser_4dt.zero_grad()
            
            logger.console_logger.info(f"[DT Training] Initial Adam optimizer for DT Agent (zero_grad())")
    
    def _reset_optimizer_4vqvae(self, logger):
        if self.main_args.optim_type.lower() == "adam":
            self.optimiser_4vqvae = Adam(params=self.mac.vqvae_model.parameters(), lr=self.main_args.vqvae_lr, weight_decay=self.main_args.weight_decay)
            self.optimiser_4vqvae.zero_grad()
            
            logger.console_logger.info(f"[VQVAE Training] Initial Adam optimizer for VqVAE Agent (zero_grad())")
            

    def zero_grad(self):
        self.pre_optimiser.zero_grad()
        self.optimiser.zero_grad()

    def update(self, pretrain=True):
        grad_norm = th.nn.utils.clip_grad_norm_(self.params, self.main_args.grad_norm_clip)
        if pretrain:
            self.pre_optimiser.step()
            self.pre_optimiser.zero_grad()
        else:
            self.optimiser.step()
            self.optimiser.zero_grad()


    def pretrain_vqvae(self, batch: EpisodeBatch, t_env: int, episode_num: int, task: str, logger):
        if self.training_steps == 0:
            self._reset_optimizer_4vqvae(logger)
            for t in self.task2args:
                task_args = self.task2args[t]
                self.task2train_info[t]["log_stats_t"] = -task_args.learner_log_interval - 1

        batch_size = batch['skill'].shape[0]
        agent_num = batch['skill'].shape[2]
        z_e_x_s, z_q_x_s, x_tilde_s= [], [], []
        self.optimiser_4vqvae.zero_grad()
        self.mac.init_hidden(batch.batch_size, task)
        for t in range(batch.max_seq_length):     
            z_e, z_q, x_tildes, hidden_states_enc, hidden_states_dec = self.mac.vqvae_model.forward(batch, t, task, 
                                                                                                    self.mac.hidden_states_enc, 
                                                                                                    self.mac.hidden_states_dec)
            self.mac.hidden_states_enc = hidden_states_enc
            self.mac.hidden_states_dec = hidden_states_dec
            x_tilde_s.append(x_tildes)
            
        x_tilde_s = th.stack(x_tilde_s, dim=1).reshape(batch_size*batch.max_seq_length*agent_num, -1)
        
        
        skill_i_label = batch['skill'].permute(0,2,1,3).reshape(batch_size*agent_num*batch.max_seq_length, -1).detach()   
        loss_recons = F.mse_loss(x_tilde_s, skill_i_label)
        loss_vq = F.mse_loss(z_q, z_e.detach())
        loss_commit = F.mse_loss(z_e, z_q.detach())
        loss = loss_recons + loss_vq + self.main_args.vqvae_beta * loss_commit
        loss.backward()
        grad_norm = th.nn.utils.clip_grad_norm_(self.params, self.main_args.grad_norm_clip)
        self.optimiser_4vqvae.step()
        self.training_steps += 1
        
        if t_env - self.task2train_info[task]["log_stats_t"] >= self.task2args[task].learner_log_interval:
            self.logger.log_stat(f"{task}/loss", loss.item(), t_env)
            self.task2train_info[task]["log_stats_t"] = t_env
            self.logger.console_logger.info("[DT Training] step={},task= {},loss={}".format(t_env,task,loss))
        

    
    
    def train_vae(self, batch: EpisodeBatch, t_env: int, episode_num: int, task: str):
        rewards = batch["reward"][:, :]
        actions = batch["actions"][:, :]
        terminated = batch["terminated"][:, :].float()
        mask = batch["filled"][:, :].float()
        mask[:, 1:] = mask[:, 1:] * (1 - terminated[:, :-1])
        avail_actions = batch["avail_actions"]

        mac_out = []
        self.mac.init_hidden(batch.batch_size, task)
        for t in range(batch.max_seq_length):
            agent_outs = self.mac.forward_skill(batch, t=t, task=task, actions=actions[:, t, :])
            mac_out.append(agent_outs)
        mac_out = th.stack(mac_out, dim=1)  # Concat over time

        seq_skill_input = F.gumbel_softmax(mac_out[:, :-self.c, :, :], dim=-1)
        kl_seq_skill = seq_skill_input * (th.log(seq_skill_input) - math.log(1 / self.main_args.skill_dim))
        enc_loss = kl_seq_skill.mean()

        dec_loss = 0.  
        self.mac.init_hidden(batch.batch_size, task)
        for t in range(batch.max_seq_length-self.c):
            seq_action_output = self.mac.forward_seq_action(batch, seq_skill_input[:, t, :, :], t, task=task)
            b, c, n, a = seq_action_output.size()
            dec_loss += (F.cross_entropy(seq_action_output.reshape(-1, a), actions[:, t:t + self.c].squeeze(-1).reshape(-1), reduction="sum") / mask[:, t:t + self.c].sum()) / n

        vae_loss = dec_loss / (batch.max_seq_length - self.c) + self.main_args.beta * enc_loss
        loss = vae_loss

        loss.backward()


        if t_env - self.task2train_info[task]["log_stats_t"] >= self.task2args[task].learner_log_interval:
            self.logger.log_stat(f"pretrain/{task}/vae_loss", vae_loss.item(), t_env)
            self.logger.log_stat(f"pretrain/{task}/enc_loss", enc_loss.item(), t_env)
            self.logger.log_stat(f"pretrain/{task}/dec_loss", dec_loss.item(), t_env)

            for i in range(self.skill_dim):
                skill_dist = seq_skill_input.reshape(-1, self.skill_dim).mean(dim=0)
                self.logger.log_stat(f"pretrain/{task}/skill_class{i+1}", skill_dist[i].item(), t_env)
            
            self.task2train_info[task]["log_stats_t"] = t_env


    def test_vae(self, batch: EpisodeBatch, t_env: int, episode_num: int, task: str):
        rewards = batch["reward"][:, :]
        actions = batch["actions"][:, :]
        terminated = batch["terminated"][:, :].float()
        mask = batch["filled"][:, :].float()
        mask[:, 1:] = mask[:, 1:] * (1 - terminated[:, :-1])
        avail_actions = batch["avail_actions"]

        mac_out = []
        self.mac.init_hidden(batch.batch_size, task)
        for t in range(batch.max_seq_length):
            agent_outs = self.mac.forward_skill(batch, t=t, task=task, actions=actions[:, t, :])
            mac_out.append(agent_outs)
        mac_out = th.stack(mac_out, dim=1)  

        seq_skill_input = F.gumbel_softmax(mac_out[:, :-self.c, :, :], dim=-1)
        kl_seq_skill = seq_skill_input * (th.log(seq_skill_input) - math.log(1 / self.main_args.skill_dim))
        enc_loss = kl_seq_skill.mean()

        dec_loss = 0.  
        self.mac.init_hidden(batch.batch_size, task)
        for t in range(batch.max_seq_length-self.c):
            seq_action_output = self.mac.forward_seq_action(batch, seq_skill_input[:, t, :, :], t, task=task)
            b, c, n, a = seq_action_output.size()
            dec_loss += (F.cross_entropy(seq_action_output.reshape(-1, a), actions[:, t:t + self.c].squeeze(-1).reshape(-1), reduction="sum") / mask[:, t:t + self.c].sum()) / n

        vae_loss = dec_loss / (batch.max_seq_length - self.c) + self.main_args.beta * enc_loss
        loss = vae_loss

        self.logger.log_stat(f"pretrain/{task}/test_vae_loss", loss.item(), t_env)
        self.logger.log_stat(f"pretrain/{task}/test_enc_loss", enc_loss.item(), t_env)
        self.logger.log_stat(f"pretrain/{task}/test_dec_loss", dec_loss.item(), t_env)

        for i in range(self.skill_dim):
            skill_dist = seq_skill_input.reshape(-1, self.skill_dim).mean(dim=0)
            self.logger.log_stat(f"pretrain/{task}/test_skill_class{i+1}", skill_dist[i].item(), t_env)

    def train_policy(self, batch: EpisodeBatch, t_env: int, episode_num: int, task: str):
        rewards = batch["reward"][:, :]
        actions = batch["actions"][:, :]
        terminated = batch["terminated"][:, :].float()
        mask = batch["filled"][:, :].float()
        mask[:, 1:] = mask[:, 1:] * (1 - terminated[:, :-1])

        with th.no_grad():
            new_actions = []
            self.mac.init_hidden(batch.batch_size, task)
            for t in range(batch.max_seq_length):
                action = self.mac.forward_skill(batch, t=t, task=task, actions=actions[:, t, :])
                label_action = action.max(dim=-1)[1].unsqueeze(-1)
                new_actions.append(label_action)
            actions = th.stack(new_actions, dim=1)

        mac_out_obs = []
        mac_out = []
        self.mac.init_hidden(batch.batch_size, task)
        for t in range(batch.max_seq_length):
            agent_outs, pri_outs = self.mac.forward_both(batch, t=t, task=task)
            mac_out.append(agent_outs)
            mac_out_obs.append(pri_outs)

        mac_out = th.stack(mac_out, dim=1)  # Concat over time
        _, _, n_agents, _ = mac_out.size()
        mac_out_obs = th.stack(mac_out_obs, dim=1)  # Concat over time
        dist_loss = F.cross_entropy(mac_out_obs.reshape(-1, self.skill_dim),
                                    actions.reshape(-1), reduction="sum") / mask.sum() / n_agents

        chosen_action_qvals = th.gather(mac_out[:, :], dim=3, index=actions[:, :]).squeeze(3)  # Remove the last dim

        target_mac_out = []
        self.target_mac.init_hidden(batch.batch_size, task)
        for t in range(batch.max_seq_length):
            target_agent_outs = self.target_mac.forward_qvalue(batch, t=t, task=task)
            target_mac_out.append(target_agent_outs)

        target_mac_out = th.stack(target_mac_out, dim=1)  # Concat across time


        # Max over target Q-Values
        if self.main_args.double_q:
            # Get actions that maximise live Q (for double q-learning)
            mac_out_detach = mac_out.clone().detach()
            # mac_out_detach[avail_actions == 0] = -9999999
            cur_max_actions = mac_out_detach[:, :].max(dim=3, keepdim=True)[1]
            target_max_qvals = th.gather(target_mac_out, 3, cur_max_actions).squeeze(3)

            cons_max_qvals = th.gather(mac_out, 3, cur_max_actions).squeeze(3)
            # cons_error = (cons_max_qvals - chosen_action_qvals).mean(dim=-1, keepdim=True)
        else:
            target_max_qvals = target_mac_out.max(dim=3)[0]
            cons_error = None

        bs, seq_len = chosen_action_qvals.size(0), chosen_action_qvals.size(1)
       
        if self.mixer is not None:
            chosen_action_qvals = self.mixer(chosen_action_qvals, batch["state"][:, :],
                                             self.task2decomposer[task])
            target_max_qvals = self.target_mixer(target_max_qvals, batch["state"][:, :],
                                                 self.task2decomposer[task])

            cons_max_qvals = self.mixer(cons_max_qvals, batch["state"][:, :],
                                        self.task2decomposer[task])

        cs_rewards = batch["reward"]
        for i in range(1, self.c):
            cs_rewards[:, :-self.c] += rewards[:, i:-(self.c - i)]

        targets = cs_rewards[:, :-self.c] + self.main_args.gamma * (1 - terminated[:, self.c - 1:-1]) * target_max_qvals[:, self.c:]

        # Td-error
        td_error = (chosen_action_qvals[:, :-self.c] - targets.detach())

        # # Cons-error
        cons_error = (cons_max_qvals - chosen_action_qvals)

        mask = mask[:, :].expand_as(cons_error)

        # 0-out the targets that came from padded data
        masked_td_error = td_error * mask[:, :-self.c]
        masked_cons_error = cons_error * mask

        # Normal L2 loss, take mean over actual data
        td_loss = (masked_td_error ** 2).sum() / mask[:, :-self.c].sum()
        cons_loss = masked_cons_error.sum() / mask.sum()
        loss = td_loss + self.alpha * cons_loss + self.phi * dist_loss

        # Do RL Learning
        self.mac.agent.encoder.requires_grad_(False)
        self.mac.agent.state_encoder.requires_grad_(False)
        self.mac.agent.decoder.requires_grad_(False)
        loss.backward()
        # episode_num should be pulic
        if (t_env - self.last_target_update_episode) / self.main_args.target_update_interval >= 1.0:
            self._update_targets()
            self.last_target_update_episode = t_env

        if t_env - self.task2train_info[task]["log_stats_t"] >= self.task2args[task].learner_log_interval:
            self.logger.log_stat(f"{task}/loss", loss.item(), t_env)
            self.logger.log_stat(f"{task}/td_loss", td_loss.item(), t_env)
            self.logger.log_stat(f"{task}/cons_loss", cons_loss.item(), t_env)
            self.logger.log_stat(f"{task}/dist_loss", dist_loss.item(), t_env)
            # self.logger.log_stat(f"{task}/grad_norm", grad_norm.item(), t_env)
            mask_elems = mask.sum().item()
            self.logger.log_stat(f"{task}/td_error_abs", (masked_td_error.abs().sum().item() / mask_elems), t_env)
            self.logger.log_stat(f"{task}/q_taken_mean", (chosen_action_qvals * mask).sum().item() / (
                        mask_elems * self.task2args[task].n_agents), t_env)
            self.logger.log_stat(f"{task}/target_mean",
                                 (targets * mask[:, :-self.c]).sum().item() / (mask_elems * self.task2args[task].n_agents), t_env)
            self.task2train_info[task]["log_stats_t"] = t_env

    def pretrain(self, batch: EpisodeBatch, t_env: int, episode_num: int, task: str):
        if self.pretrain_steps == 0:
            self._reset_optimizer()
            for t in self.task2args:
                task_args = self.task2args[t]
                self.task2train_info[t]["log_stats_t"] = -task_args.learner_log_interval - 1
        
        self.train_vae(batch, t_env, episode_num, task)
        self.pretrain_steps += 1
    
    def test_pretrain(self, batch: EpisodeBatch, t_env: int, episode_num: int, task: str):
        self.test_vae(batch, t_env, episode_num, task)
    
    
    
    def train(self, batch: EpisodeBatch, t_env: int, episode_num: int, task: str):
        if self.training_steps == 0:
            self._reset_optimizer()
            for t in self.task2args:
                task_args = self.task2args[t]
                self.task2train_info[t]["log_stats_t"] = -task_args.learner_log_interval - 1

        self.train_policy(batch, t_env, episode_num, task)
        self.training_steps += 1

    def _update_targets(self):
        self.target_mac.load_state(self.mac)
        if self.mixer is not None:
            self.target_mixer.load_state_dict(self.mixer.state_dict())
        self.logger.console_logger.info("Updated target network")

    def cuda(self):
        self.mac.cuda()
        self.target_mac.cuda()
        if self.mixer is not None:
            self.mixer.cuda()
            self.target_mixer.cuda()

    def save_models(self, path):
        if self.main_args.train_DT:
            self.mac.save_models(path)
        
        elif self.main_args.pretrain_vqvae:
            self.mac.save_models(path)
        
        elif self.main_args.train_DT_w_glsk:
            self.mac.save_models(path)
            
        else:
            self.mac.save_models(path)
            if self.mixer is not None:
                th.save(self.mixer.state_dict(), "{}/mixer.th".format(path))
            th.save(self.optimiser.state_dict(), "{}/opt.th".format(path))

    def load_models(self, path):
        self.mac.load_models(path)
        self.target_mac.load_models(path)
        if self.mixer is not None:
            self.mixer.load_state_dict(th.load("{}/mixer.th".format(path), map_location=lambda storage, loc: storage))
        
    
    
    def train_by_skill(self, batch: EpisodeBatch, t_env: int, episode_num: int, task: str, logger, task2decomposer):
        if self.training_steps == 0:
            self._reset_optimizer_4dt(logger)
            for t in self.task2args: # t: task
                task_args = self.task2args[t]
                self.task2traindt_info[t]["log_stats_t"] = -task_args.learner_log_interval - 1
        
        T = batch.data['filled'].shape[1]
        for _ in range(self.main_args.train_steps_per_batch):
        
            t = random.randint(1, T-1)
            
            step_start = max(0, t+1 - self.main_args.step_block_size)
            step_end = t + 1
            right_padding_num = self.main_args.step_block_size - (step_end - step_start)
            assert right_padding_num >= 0
                
            input, mask_on_valid_steps = self.get_batch(batch, task2decomposer, task, step_start, step_end, right_padding_num)
            
            self.train_step(input, task, t_env, mask_on_valid_steps)
            self.training_steps += 1
        
    
    def get_batch(self, batch, task2decomposer, task, step_start, step_end, right_padding_num):

        batch_size = batch['skill'].shape[0]
        T          = batch['skill'].shape[1]
        agent_num  = batch['skill'].shape[2]
        
          
        obs_input = self.mac._build_inputs_4dt(batch, step_start, step_end, task).reshape(batch_size, step_end-step_start, agent_num, -1)      
        obs_input = th.cat([obs_input, th.zeros(batch_size, right_padding_num, *obs_input.shape[2:]).to('cuda')], dim=1)\
                                                                    .reshape(batch_size*(step_end-step_start+right_padding_num)*agent_num,-1) 
                                                                    
        sk = deepcopy(batch.data['skill'][:,step_start:step_end,:,:])                                                       
        sk = th.cat([sk, th.zeros(batch_size, right_padding_num, agent_num, self.main_args.skill_dim).to('cuda')], dim=1)    
        
        gl_sk_id = deepcopy(batch.data['global_skill_id'][:,step_start:step_end,:,:])                                       
        gl_sk_id = th.cat([gl_sk_id, th.zeros(batch_size, right_padding_num, agent_num, 1).to('cuda')], dim=1)               
        
        rtgs = self.compute_rtg(batch['reward'], gamma=self.main_args.gamma).unsqueeze(-2).repeat(1,1,agent_num,1)                                     
        rtg = rtgs[:,step_start:step_end,:,:]                                                                              
        rtg = th.cat([rtg, th.zeros(batch_size, right_padding_num, agent_num, 1).to('cuda')], dim=1)                         
        
        
        # 2.mask_on_valid_steps
        terminated_step = th.zeros(batch_size, dtype=th.int64)
        for i in range(batch_size):
            terminated_step[i] = th.nonzero(batch['terminated'][i] == 1).squeeze()[0]
        mask_on_valid_steps = th.zeros_like(batch['terminated'])
        for i in range(batch_size):
            mask_on_valid_steps[i, :terminated_step[i]+1, :] = 1                                                                    
        mask_on_valid_steps = mask_on_valid_steps.unsqueeze(-1).repeat(1,1,agent_num,1)                                              
        mask_on_valid_steps = mask_on_valid_steps[:, step_start:step_end, :]                                                         
        mask_on_valid_steps = th.cat([mask_on_valid_steps, th.zeros(batch_size, right_padding_num, agent_num, 1).to('cuda')], dim=1) 
        
        input4dt = (obs_input, sk, gl_sk_id, rtg, step_start, step_end+right_padding_num)
        
        return input4dt, mask_on_valid_steps 
        
    def train_step(self, input, task, t_env, mask_on_valid_steps):

        if mask_on_valid_steps.sum() <= 20:
            return
        
        obs_input, sk, gl_sk_id, rtg, step_start, step_end = input
        skill_label = deepcopy(sk).detach().reshape(-1, sk.shape[-1]) 
        gl_sk_id_label =  deepcopy(gl_sk_id).detach().type(th.int64).reshape(-1, gl_sk_id.shape[-1]).squeeze(-1) 
        
        with th.set_grad_enabled(True):
            gl_sk_id, skill_type = self.mac.dt_agent(input, task)
            
            criterion1 = nn.MSELoss(reduction='none')
            skill_loss = (criterion1(skill_type.reshape(-1, skill_type.size(-1)), skill_label) * mask_on_valid_steps.reshape(-1, 1)).sum()
            
            criterion2 = nn.CrossEntropyLoss(reduction='none')
            gl_sk_id_loss = (criterion2(gl_sk_id.reshape(-1, gl_sk_id.size(-1)), gl_sk_id_label).unsqueeze(-1) * mask_on_valid_steps.reshape(-1, 1)).sum()
            
            loss = skill_loss + gl_sk_id_loss * self.main_args.codebook_alpha

            loss = loss / mask_on_valid_steps.sum()
            assert mask_on_valid_steps.sum()>=1
            
            
        self.optimiser_4dt.zero_grad()
        loss.backward()
        th.nn.utils.clip_grad_norm_(self.mac.dt_agent.parameters(), self.main_args.grad_norm_clip)
        self.optimiser_4dt.step()
        
        if t_env - self.task2train_info[task]["log_stats_t"] >= self.task2args[task].learner_log_interval:
            self.logger.log_stat(f"{task}/loss", loss.item(), t_env)
            self.logger.log_stat(f"{task}/skill_loss", skill_loss.item(), t_env)
            self.logger.log_stat(f"{task}/gl_sk_id_loss", gl_sk_id_loss.item(), t_env)
            self.task2train_info[task]["log_stats_t"] = t_env
            self.logger.console_logger.info("[DT Training] step={},task= {},loss={}, skill_loss={}, gl_sk_id_loss={}".format(t_env,task,loss, skill_loss, gl_sk_id_loss))
    
    
    

    def train_DTpolicy(self, batch: EpisodeBatch, 
                       t_env: int, 
                       input,
                       skill_label,
                       episode_num: int, 
                       task: str, 
                       step_start: int, step_end: int):
                
        with th.set_grad_enabled(True):
            logits_skill = self.mac.dt_agent(input, task)
            
            criterion = nn.MSELoss()
        
            loss = criterion(logits_skill.reshape(-1, logits_skill.size(-1)), skill_label)
            entropy_info = 0.
            ratio_info = 0.
            confidence_info = 0.
            
        self.optimiser_4dt.zero_grad()
        loss.backward()
        th.nn.utils.clip_grad_norm_(self.mac.dt_agent.parameters(), self.main_args.grad_norm_clip)
        self.optimiser_4dt.step()
        
        if t_env - self.task2train_info[task]["log_stats_t"] >= self.task2args[task].learner_log_interval:
            self.logger.log_stat(f"{task}/loss", loss.item(), t_env)
            self.task2train_info[task]["log_stats_t"] = t_env
            self.logger.console_logger.info("[DT Training] step={},task= {},loss={}".format(t_env,task,loss))

    def compute_rtg(self, reward, gamma=1):
        
        rtg = th.zeros_like(reward)
        running_sum = 0.0
        for t in reversed(range(reward.shape[1])):
            running_sum = reward[:,t] + gamma * running_sum
            rtg[:,t] = running_sum
        return rtg

